const ITERATOR_METHODS = require('../open-api-client/constants/iterator-methods')
const MUTED_ERROR_CODES = require('../open-api-client/constants/muted-error-codes')
const openAPIClient = require('../open-api-client/open-api-client')
const { defaults } = require('@kl/constants')
const BaseConnection = require('../connection-pool/base-connection')

module.exports = class OpenAPIServerTransport {
  constructor ({ getConnection, logIncomigActivity = uid => { } }) {
    this.init = init

    function init ({ transport: { on, send, uid } }) {
      if (typeof on !== 'function') {
        throw new Error('Open API server transport must have "on()" method')
      }
      if (typeof send !== 'function') {
        throw new Error('Open API server transport must have "send()" method')
      }
      if (typeof uid !== 'string') {
        throw new Error('Open API server transport must have string "uid"')
      }
      const transport = { on, send, uid }
      transport.on('run-command', function (command, data, cacheConfig) {
        logIncomigActivity(transport.uid)
        onCommandStart(transport, command, data, cacheConfig)
      })
      transport.on('open-api:get-interface', function () {
        logIncomigActivity(transport.uid)
        transport.send('open-api:announce-interface', openAPIClient.interface)
      })
    }

    function onCommandEnd (transport, command, resultData) {
      const data = resultData?.result || null
      const maxChunkSize = defaults.defaultChunkSize
      if (!Buffer.isBuffer(data) || data.length < maxChunkSize) {
        command.result = resultData
        command.result.itr = isIterator(command.name)
        transport.send('command-end', command)
      } else {
        const fullDataLength = data.length
        let sentDataLength = 0
        let remainingFileLength = fullDataLength

        while (remainingFileLength > 0) {
          const chunkData = data.slice(sentDataLength, sentDataLength + maxChunkSize)
          const chunkLength = chunkData.length
          remainingFileLength -= chunkLength

          command.chunk = {
            data: chunkData,
            from: sentDataLength,
            to: sentDataLength + chunkLength,
            fullDataLength,
            isLast: remainingFileLength <= 0
          }
          transport.send('command-chunk', command)
          sentDataLength = sentDataLength + chunkLength
        }
      }
    }

    function onCommandError (transport, command, error) {
      runtime.logger.error(`Error while processing command: ${command}`, error)
      command.error = getErrorMessage(error)
      command.errorLocalization = error.localization
      transport.send('command-error', command)
    }

    function getErrorMessage (error) {
      if (!error) {
        return 'No error passed'
      }

      if (MUTED_ERROR_CODES.includes(error.code)) {
        return error.code
      }

      if (error instanceof BaseConnection) {
        const { sessionId, clientUid } = error
        return `Session "${sessionId}" has expired for clientUid=${clientUid}`
      }

      return error.message ? error.message : error
    }

    function isIterator (cmdName) {
      return ITERATOR_METHODS.includes(cmdName)
    }

    async function runCommand (transport, command, data, cacheConfig) {
      const successCallback = onCommandEnd.bind(this, transport, command)
      const errorCallback = onCommandError.bind(this, transport, command)

      const [openAPIClass, openAPIMethod] = command.name.split('.')

      if (!openAPIClient.interface[openAPIClass] || !openAPIClient.interface[openAPIClass][openAPIMethod]) {
        const error = `There is no ${openAPIClass}.${openAPIMethod} command in Open API interface`
        errorCallback(error)
        return
      };

      const connection = getConnection({
        uid: transport.uid,
        ...command
      })
      if (!connection) {
        onCloseConnection(new Error('Connection not found'))
        errorCallback('Connection not found')
        return
      } else if (!connection.isEventHandled('error')) {
        connection.on('error', onCloseConnection)
      }

      try {
        const response = await openAPIClient[openAPIClass][openAPIMethod](data, connection, cacheConfig)
        successCallback(response)
      } catch (error) {
        errorCallback(error)
      }

      function onCloseConnection (e) {
        runtime.logger.error(getErrorMessage(e))
        runtime.logger.log('API connection is lost. Logout.')
        transport.send('logout')
      }
    }

    function onCommandStart (transport, command, data, cacheConfig) {
      let cmdText = 'ChunkAccessor.'
      if (command.name === 'itr') {
        const start = data.start
        switch (data.action) {
          case 'next':
            cmdText += 'GetItemsChunk'
            break
          case 'prev':
            cmdText += 'GetItemsChunk'
            break
          case 'close':
            cmdText += 'Release'
            break
          case 'count':
            cmdText += 'GetItemsCount'
            break
        }
        command.name = cmdText
        runCommand(transport, command, {
          strAccessor: data.id,
          nStart: start,
          nCount: data.count
        })
      } else {
        runCommand(transport, command, data, cacheConfig)
      }
    }
  }
}
