"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAccessToken = exports.getConnection = void 0;
const dotenv = require("dotenv");
const path = require("path");
const axios_1 = require("axios");
const openapi_client_1 = require("@kl/openapi-client");
dotenv.config({ path: path.join(__dirname.split('node_modules')[0], '.env') });
const getConnectionVariables = () => {
    const { XDR_KSC_PROXY_HOST, XDR_KSC_PROXY_PORT, IAM_HOST, IAM_PORT, XDR_KSC_PROXY_VERSION, OAUTH_CLIENT_ID_FILE, OAUTH_CLIENT_SECRET_FILE } = process.env;
    if (!(XDR_KSC_PROXY_HOST &&
        XDR_KSC_PROXY_PORT &&
        IAM_HOST &&
        IAM_PORT &&
        XDR_KSC_PROXY_VERSION &&
        OAUTH_CLIENT_ID_FILE &&
        OAUTH_CLIENT_SECRET_FILE)) {
        throw Error('Missing environment variables to execute openApi methods');
    }
    return {
        XDR_KSC_PROXY_HOST,
        XDR_KSC_PROXY_PORT,
        IAM_HOST,
        IAM_PORT,
        XDR_KSC_PROXY_VERSION,
        OAUTH_CLIENT_ID_FILE,
        OAUTH_CLIENT_SECRET_FILE
    };
};
(0, openapi_client_1.init)();
const getConnection = async (accessToken, idsKsc) => {
    try {
        const { XDR_KSC_PROXY_HOST, XDR_KSC_PROXY_VERSION, XDR_KSC_PROXY_PORT } = getConnectionVariables();
        const createConnectionParams = {
            openAPIHost: XDR_KSC_PROXY_HOST,
            openAPIPort: XDR_KSC_PROXY_PORT,
            openAPIVersion: XDR_KSC_PROXY_VERSION,
            oAuthToken: {
                access_token: ''
            }
        };
        const connectionId = await openapi_client_1.connectionPool.createConnection(createConnectionParams, (0, openapi_client_1.getConnectionConstructorByAuthType)('oAuthToken'));
        const connection = openapi_client_1.connectionPool.getConnection(connectionId);
        connection.customHeaders = Object.assign(Object.assign({}, connection.customHeaders), { Authorization: `Bearer ${accessToken}`, KLSRVH_SRV_INST_ID: `${idsKsc.instanceId}/${idsKsc.virtualServerId}` });
        return connection;
    }
    catch (error) {
        throw Error(`Get connection error: ${error.message}`);
    }
};
exports.getConnection = getConnection;
const getAccessToken = async (jwt) => {
    try {
        const { IAM_HOST, IAM_PORT, OAUTH_CLIENT_ID_FILE, OAUTH_CLIENT_SECRET_FILE } = getConnectionVariables();
        const url = `https://${IAM_HOST}:${IAM_PORT}/oauth2/token`;
        const { data } = await axios_1.default.post(url, new URLSearchParams({
            client_id: OAUTH_CLIENT_ID_FILE,
            client_secret: OAUTH_CLIENT_SECRET_FILE,
            grant_type: 'urn:ietf:params:oauth:grant-type:jwt-bearer',
            assertion: jwt.accessTokens.IAM,
            scope: 'kscsession'
        }));
        return data.access_token;
    }
    catch (error) {
        throw Error(`Token exchange error: ${error.message}`);
    }
};
exports.getAccessToken = getAccessToken;
