"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMessageHandler = void 0;
const nats_1 = require("nats");
const services_1 = require("../../services");
const serializers = require("./serializers/interpreter");
const jsonCodec = (0, nats_1.JSONCodec)();
const getMessageHandler = (subject, actions, isAdditive) => async ({ reply, data }, connection) => {
    if (!connection) {
        runtime.logger.error(`${services_1.LOGS_PREFIX}${subject} error. There is no NATS connection`);
        return;
    }
    if (!reply) {
        runtime.logger.error(`${services_1.LOGS_PREFIX}${subject} error. There is no "reply" field in msg`);
        return;
    }
    const handleMessageError = (errorMessage) => {
        runtime.logger.error(`${services_1.LOGS_PREFIX}${subject} error. ${errorMessage}`);
        connection.publish(reply, jsonCodec.encode({
            description: errorMessage,
            resultCode: 'fail',
            errorCode: 'unsupported'
        }));
    };
    let decodedData = null;
    try {
        decodedData = jsonCodec.decode(data);
    }
    catch (error) {
        handleMessageError(`Decode data. ${error}`);
        return;
    }
    if (!decodedData.version) {
        handleMessageError('There is no version in message');
        return;
    }
    const serializerVersion = serializers[decodedData.version];
    if (!serializerVersion) {
        handleMessageError(`There is no serializer for message version "${decodedData.version}"`);
        return;
    }
    const responseFunction = actions;
    const action = responseFunction === null || responseFunction === void 0 ? void 0 : responseFunction[decodedData.action];
    if (!action) {
        const errorMessage = handleMessageError(`Response function has no "${decodedData.action}" action`);
        connection.publish(reply, jsonCodec.encode({ error: errorMessage }));
        return;
    }
    const serializer = isAdditive
        ? serializerVersion.getResponseFunctionAdditiveActionArgs
        : serializerVersion.getResponseFunctionActionArgs;
    const args = serializer(decodedData);
    if ('errorMessage' in args) {
        handleMessageError(`Incorrect parameter: ${args.errorMessage}`);
        return;
    }
    const result = await action(...args);
    connection.publish(reply, jsonCodec.encode(result));
};
exports.getMessageHandler = getMessageHandler;
