"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getResponseTaskApi = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const fs = require("fs");
const constants_1 = require("./constants");
const openapi_1 = require("../../openapi");
const openapi_client_1 = require("@kl/openapi-client");
const bucket_1 = require("../../services/nats/bucket");
const bucket = new bucket_1.NatsBucket();
const getResponseTaskApi = () => {
    return {
        async calculateIdempotentTaskUid(ctx, taskType) {
            if (ctx && taskType) {
                return {
                    resultCode: 'ok',
                    taskUid: (0, crypto_1.createHash)('sha1')
                        .update(JSON.stringify(Object.assign(Object.assign({}, taskType), { responseId: ctx.responseId })))
                        .digest('hex')
                };
            }
            else {
                return {
                    resultCode: 'fail',
                    taskUid: ''
                };
            }
        },
        async runResponseTask(ctx, sctx, taskType, targetHosts, taskSettings) {
            var _a, _b;
            const webFolderPath = __dirname.split('node_modules')[0];
            const globalConfigPath = (0, path_1.resolve)(webFolderPath, 'assets', 'task-types.json');
            const typesTasks = JSON.parse(await fs.promises.readFile(globalConfigPath, 'utf-8'));
            const { taskUid } = await this.calculateIdempotentTaskUid(ctx, taskType);
            const taskName = `${constants_1.PREFIX_TASK_NAME}-${taskUid}`;
            try {
                const accessToken = await (0, openapi_1.getAccessToken)(sctx);
                const connection = await (0, openapi_1.getConnection)(accessToken, ctx.extra.ksc);
                const vServer = await openapi_client_1.openAPIClient.HostGroup.GetStaticInfo(null, connection);
                const isSupportVersionSetUniqueId = checkValidVersionKsc(vServer.result.KLADMSRV_PRODUCT_FULL_VERSION, constants_1.VERSION_KSC_SUPPORTED_SET_UNIQUE_ID);
                const id = await bucket.getTaskId(taskUid);
                if (id) {
                    throw Error(`Task with task name ${taskName} already exists)`);
                }
                const requireData = {
                    taskData: Object.assign(Object.assign({}, constants_1.DEFAULT_TASK_PARAMS), { TASK_UNIQUE_ID: isSupportVersionSetUniqueId ? taskUid : undefined, TASKID_COMPONENT_NAME: (_a = typesTasks.find(task => task.type === taskType.taskType)) === null || _a === void 0 ? void 0 : _a.CID_ComponentID, TASK_NAME: taskType.taskType, TASKID_PRODUCT_NAME: taskType.productId, TASKID_VERSION: taskType.productVersion, TASK_ADDITIONAL_PARAMS: Object.assign(Object.assign({}, taskSettings), { TASK_AUTOCREATED_GRP: 'IRP', KLPRCI_TASK_INVISIBLE: !process.env.VISIBLE_RESPONSIVE_TASK }), TASK_INFO_PARAMS: Object.assign(Object.assign({}, constants_1.DEFAULT_TASK_PARAMS.TASK_INFO_PARAMS), { DisplayName: taskName, HostList: transformParamsListContainers(targetHosts) }), TASKSCH_FIRST_EXECUTION_TIME: {
                            type: 'datetime',
                            value: `${new Date().toISOString().split('.')[0]}Z`
                        } })
                };
                runtime.logger.log(`KSCApi Request from plugin-sdk: ${JSON.stringify(requireData)}`);
                const { result } = (await openapi_client_1.openAPIClient.Tasks.AddTask(requireData, connection));
                if ('PxgRetVal' in result) {
                    await bucket.setTaskId(taskUid, result.PxgRetVal);
                    return {
                        taskName,
                        taskUid: result.PxgRetVal,
                        resultCode: 'ok'
                    };
                }
                else {
                    return {
                        taskName,
                        taskUid: '',
                        internalCode: result.PxgError.code.toString(),
                        resultCode: 'fail',
                        description: result.PxgError.message
                    };
                }
            }
            catch (error) {
                runtime.logger.error(JSON.stringify(error));
                return {
                    taskName: taskName,
                    taskUid: '',
                    resultCode: 'fail',
                    errorCode: 'internalError',
                    description: (_b = error.message) !== null && _b !== void 0 ? _b : error
                };
            }
        },
        async addHostToTask(ctx, sctx, taskUid, targetHost) {
            var _a, _b;
            try {
                if (!((_a = targetHost.integrationSpecific) === null || _a === void 0 ? void 0 : _a.ksc))
                    throw new Error('Target host is not defined');
                const accessToken = await (0, openapi_1.getAccessToken)(sctx);
                const connection = await (0, openapi_1.getConnection)(accessToken, ctx.extra.ksc);
                const vServer = await openapi_client_1.openAPIClient.HostGroup.GetStaticInfo(null, connection);
                const isSupportVersion = checkValidVersionKsc(vServer.result.KLADMSRV_PRODUCT_FULL_VERSION, constants_1.VERSION_KSC_SUPPORTED_METHOD_ADD_HOST_TO_TASK);
                if (!isSupportVersion) {
                    return {
                        resultCode: 'fail',
                        errorCode: 'unsupported',
                        description: 'This method is not supported in this version of ksc.'
                    };
                }
                const strTaskId = await openapi_client_1.openAPIClient.Tasks.ResolveTaskId({ strPrtsTaskId: taskUid }, connection);
                if (typeof strTaskId.result !== 'string')
                    throw new Error(strTaskId.result.PxgError.message);
                const resultAddHostToTask = await openapi_client_1.openAPIClient.Tasks.AddHostToTask({
                    strTask: strTaskId.result,
                    strHostName: targetHost.integrationSpecific.ksc
                }, connection);
                if (resultAddHostToTask.result.PxgError) {
                    throw new Error(resultAddHostToTask.result.PxgError.message);
                }
                return {
                    resultCode: 'ok'
                };
            }
            catch (error) {
                return {
                    resultCode: 'fail',
                    errorCode: 'internalError',
                    description: (_b = error.message) !== null && _b !== void 0 ? _b : error
                };
            }
        },
        async getTaskStatistics(ctx, sctx, taskUid) {
            var _a;
            try {
                const accessToken = await (0, openapi_1.getAccessToken)(sctx);
                const connection = await (0, openapi_1.getConnection)(accessToken, ctx.extra.ksc);
                const taskId = await getTaskId(taskUid, connection);
                const taskStatistics = await openapi_client_1.openAPIClient.Tasks.GetTaskStatistics(taskId, connection);
                if (taskStatistics.result.PxgError)
                    throw new Error(taskStatistics.result.PxgError.message);
                const statistics = [];
                for (const taskState in taskStatistics.result) {
                    if (Number(taskState)) {
                        statistics.push({
                            taskState: Number(taskState),
                            count: taskStatistics.result[Number(taskState)]
                        });
                    }
                }
                return {
                    resultCode: 'ok',
                    statistics
                };
            }
            catch (error) {
                runtime.logger.error(JSON.stringify(error));
                return {
                    resultCode: 'fail',
                    statistics: [],
                    description: (_a = error.message) !== null && _a !== void 0 ? _a : error,
                    errorCode: 'internalError'
                };
            }
        },
        async getTaskStates(ctx, sctx, taskUid, targetHosts = [], interestedStates = []) {
            var _a;
            try {
                const accessToken = await (0, openapi_1.getAccessToken)(sctx);
                const connection = await (0, openapi_1.getConnection)(accessToken, ctx.extra.ksc);
                let interestedStatesList;
                if (!(interestedStates.length && interestedStates.every(el => constants_1.TASK_STATUSES.includes(el)))) {
                    interestedStatesList = constants_1.TASK_STATUSES;
                }
                else {
                    interestedStatesList = interestedStates;
                }
                const taskId = await getTaskId(taskUid, connection);
                let hostsId;
                if (!targetHosts.length) {
                    const { result } = await openapi_client_1.openAPIClient.Tasks.GetTaskData({ taskId }, connection);
                    if (result.PxgError)
                        throw new Error(result.PxgError.message);
                    hostsId = result.TASK_INFO_PARAMS.HostList.map((el) => el.HostName);
                }
                else
                    hostsId = targetHosts.map(el => { var _a; return (_a = el.integrationSpecific) === null || _a === void 0 ? void 0 : _a.ksc; });
                const res = {
                    statuses: []
                };
                for (const status of interestedStatesList) {
                    const strHostIteratorId = await openapi_client_1.openAPIClient.Tasks.ResetHostIteratorForTaskStatus({
                        strTask: taskId,
                        pFields2Return: constants_1.FIELD_RETURN_TASK_STATUS,
                        nHostStateMask: status,
                        nLifetime: 1
                    }, connection);
                    if (typeof strHostIteratorId.result !== 'string')
                        throw new Error(strHostIteratorId.result.PxgError.message);
                    const resultCount = (await openapi_client_1.openAPIClient.Tasks.GetHostStatusRecordsCount(strHostIteratorId.result, connection));
                    const statusHostTasks = await openapi_client_1.openAPIClient.Tasks.GetHostStatusRecordRange({
                        strHostIteratorId: strHostIteratorId.result,
                        nStart: 0,
                        nEnd: 'PxgRetVal' in resultCount.result ? resultCount.result.PxgRetVal : 0
                    }, connection);
                    openapi_client_1.openAPIClient.SrvView.ReleaseIterator({ wstrIteratorId: strHostIteratorId.result }, connection);
                    if (statusHostTasks.result && 'PxgError' in statusHostTasks.result) {
                        throw new Error(statusHostTasks.result.PxgError.message);
                    }
                    statusHostTasks.result &&
                        res.statuses.push(...statusHostTasks.result.statuses.filter(stat => hostsId.includes(stat.value.hostname)));
                }
                if (res.statuses.length) {
                    return {
                        resultCode: 'ok',
                        states: res.statuses.map(status => {
                            var _a, _b, _c;
                            return {
                                asset: { uid: status.value.hostname },
                                progressPercent: status.value.completed_percent,
                                state: status.value.state_code,
                                descr: status.value.state_descr,
                                eventDbId: (_a = status.value.src_event_id) === null || _a === void 0 ? void 0 : _a.value,
                                startDateTimeISO: (_b = status.value.start_time) === null || _b === void 0 ? void 0 : _b.value,
                                finishDateTimeISO: (_c = status.value.finish_time) === null || _c === void 0 ? void 0 : _c.value
                            };
                        })
                    };
                }
                else {
                    return {
                        resultCode: 'ok',
                        states: [],
                        description: 'Target hosts and interested states not found.'
                    };
                }
            }
            catch (error) {
                runtime.logger.error(JSON.stringify(error));
                return {
                    resultCode: 'fail',
                    states: [],
                    description: (_a = error.message) !== null && _a !== void 0 ? _a : error,
                    errorCode: 'internalError'
                };
            }
        },
        async getTaskEvents(ctx, sctx, taskUid, targetHosts = [], filter = '') {
            var _a;
            try {
                const accessToken = await (0, openapi_1.getAccessToken)(sctx);
                const connection = await (0, openapi_1.getConnection)(accessToken, ctx.extra.ksc);
                const strTask = await getTaskId(taskUid, connection);
                const summaryFilter = `(& (| ${targetHosts.map(host => `(hostname = "${host.uid}")`).join(' ')}) ${filter})`;
                const resultHistory = await openapi_client_1.openAPIClient.Tasks.GetTaskHistory({
                    strTask,
                    pFields2Return: constants_1.FIELD_RETURN_TASK_HISTORY,
                    strHostName: '',
                    pSortFields: [
                        {
                            Name: 'rise_time',
                            Asc: false
                        }
                    ],
                    pFilter: {
                        EVP_INCL_GNRL_EVENTS: false,
                        EVP_INCL_TASK_STATES: true,
                        KLEVP_RFC2254_FILTER: summaryFilter
                    }
                }, connection);
                if ('PxgError' in resultHistory.result)
                    throw new Error(resultHistory.result.PxgError.message);
                const resultCount = await openapi_client_1.openAPIClient.EventProcessing.GetRecordCount({
                    strIteratorId: resultHistory.result.strIteratorId
                }, connection);
                if (typeof resultCount.result !== 'number')
                    throw new Error(resultCount.result.PxgError.message);
                const resultRange = await openapi_client_1.openAPIClient.EventProcessing.GetRecordRange({
                    strIteratorId: resultHistory.result.strIteratorId,
                    nStart: 0,
                    nEnd: Number(resultCount.result)
                }, connection);
                if ('PxgError' in resultRange.result)
                    throw new Error(resultRange.result.PxgError.message);
                openapi_client_1.openAPIClient.SrvView.ReleaseIterator({ wstrIteratorId: resultHistory.result.strIteratorId }, connection);
                return {
                    resultCode: 'ok',
                    events: resultRange.result.KLEVP_EVENT_RANGE_ARRAY
                };
            }
            catch (error) {
                runtime.logger.error(JSON.stringify(error));
                return {
                    resultCode: 'fail',
                    description: (_a = error.message) !== null && _a !== void 0 ? _a : error,
                    errorCode: 'internalError',
                    events: []
                };
            }
        },
        async deleteTask(ctx, sctx, taskUid, forceForIncomplete) {
            var _a;
            try {
                const accessToken = await (0, openapi_1.getAccessToken)(sctx);
                const connection = await (0, openapi_1.getConnection)(accessToken, ctx.extra.ksc);
                const strTask = await getTaskId(taskUid, connection);
                const { result } = await openapi_client_1.openAPIClient.Tasks.GetTaskStatistics(strTask, connection);
                if ('PxgError' in result)
                    throw new Error(result.PxgError.message);
                const isIncomplete = constants_1.INCOMPLETE_STATUSES.some(statusId => !!result[statusId]);
                if (forceForIncomplete || !isIncomplete) {
                    const statusDeleteTask = await openapi_client_1.openAPIClient.Tasks.DeleteTask({ strTask }, connection);
                    if ('PxgError' in statusDeleteTask.result)
                        throw new Error(statusDeleteTask.result.PxgError.message);
                    await bucket.deleteKeyValue(taskUid);
                    return { resultCode: 'ok' };
                }
                else {
                    return { resultCode: 'retry', description: 'The task is executing on the target hosts' };
                }
            }
            catch (error) {
                runtime.logger.error(JSON.stringify(error));
                return {
                    resultCode: 'fail',
                    description: (_a = error.message) !== null && _a !== void 0 ? _a : error,
                    errorCode: 'internalError'
                };
            }
        }
    };
};
exports.getResponseTaskApi = getResponseTaskApi;
const transformParamsListContainers = (targetHosts) => {
    return targetHosts.map(el => { var _a; return ({ HostName: (_a = el.integrationSpecific) === null || _a === void 0 ? void 0 : _a.ksc }); });
};
const checkValidVersionKsc = (versionKsc, supportedVersion) => {
    const versionKscList = versionKsc.split('.').map(el => Number(el));
    const supportedVersionList = supportedVersion.split('.').map(el => Number(el));
    for (let i = 0; i <= versionKscList.length; i++) {
        if (supportedVersionList[i] > versionKscList[i]) {
            return false;
        }
        else if (supportedVersionList[i] < versionKscList[i]) {
            return true;
        }
    }
    return true;
};
const getTaskId = async (taskUid, connection) => {
    const id = await bucket.getTaskId(taskUid);
    if (id) {
        return id;
    }
    const strTaskId = await openapi_client_1.openAPIClient.Tasks.ResolveTaskId({ strPrtsTaskId: taskUid }, connection);
    if (typeof strTaskId.result === 'string') {
        return strTaskId.result;
    }
    throw new Error(`Task with task id=${taskUid} not found.`);
};
