const DataStore = require('./src/data-store')
const { promisify } = require('util')
const redis = require('redis')
const {
  encrypt,
  decrypt
} = require('./src/processing')
const {
  REDIS_HOST = '127.0.0.1',
  REDIS_PORT = 6379,
  REDIS_RECONNECT_TIMEOUT = 1000,
  REDIS_RECONNECT_ATTEMPTS
} = process.env

module.exports = class RedisDataStore extends DataStore {
  constructor () {
    super(...arguments)
    const client = redis.createClient({
      host: REDIS_HOST,
      port: Number(REDIS_PORT),
      retry_strategy: ({ attempt }) => {
        if (REDIS_RECONNECT_ATTEMPTS && attempt > Number(REDIS_RECONNECT_ATTEMPTS)) {
          return new Error('Error: Data-store: Cannot connect to Redis. Maximum connection attempts reached')
        }

        console.error('Warning: Data-store: Redis connection is lost. Attempting reconnect')
        return Number(REDIS_RECONNECT_TIMEOUT)
      }
    })
    this.store = {
      get: promisify(client.get).bind(client),
      set: promisify(client.set).bind(client),
      delete: promisify(client.del).bind(client),
      pushToList: promisify(client.rpush).bind(client),
      getList: promisify(client.lrange).bind(client),
      removeFromListByValue: promisify(client.lrem).bind(client),
      setToHashMap: promisify(client.hset).bind(client),
      removeFromHashMap: promisify(client.hdel).bind(client),
      getHashMap: promisify(client.hgetall).bind(client),
      getFromHashMap: promisify(client.hget).bind(client)
    }
  }

  async clear () {
    if (!this.keys.length) return
    await this.store.delete(...this.keys)
    this.keys = []
  }

  packValue (value) {
    const packedValue = super.packValue(value)
    return packedValue && encrypt(packedValue)
  }

  unpackValue (value) {
    const decryptedValue = value && decrypt(value)
    return super.unpackValue(decryptedValue)
  }

  pushToList (name, value) {
    this.validateKey(name)
    if (!this.keys.includes(name)) this.keys.push(name)
    return this.store.pushToList(this.keyPrefix + name, this.packValue(value))
  }

  setToHashMap (name, key, value) {
    this.validateKey(name)
    if (!this.keys.includes(name)) this.keys.push(name)
    return this.store.setToHashMap(this.keyPrefix + name, key, this.packValue(value))
  }

  async getFromHashMap (name, key) {
    const value = await this.store.getFromHashMap(this.keyPrefix + name, key)
    return this.unpackValue(value)
  }

  removeFromListByValue (name, value) {
    return this.store.removeFromListByValue(this.keyPrefix + name, 0, this.packValue(value))
  }

  removeFromHashMap (name, key) {
    return this.store.removeFromHashMap(this.keyPrefix + name, key)
  }

  async getList (name) {
    const list = await this.store.getList(this.keyPrefix + name, 0, -1)
    return list.map(value => this.unpackValue(value))
  }

  async getHashMap (name) {
    const keysValues = await this.store.getHashMap(this.keyPrefix + name) || {}

    return Object.entries(keysValues).reduce((acc, [key, value]) => {
      acc[key] = this.unpackValue(value)

      return acc
    }, {})
  }
}
