/**
 * Data store factory class.
 * Each data store instance is kept under unique user ID
 */
const InMemoryDataStore = require('../in-memory')
const RedisDataStore = require('../redis')

module.exports = class DataStoreFactory {
  constructor ({ keyPrefix = '' } = {}) {
    this.context = new Map()
    this.keyPrefix = keyPrefix
  }

  /**
   * Resolves constructor for gitven data-store type
   * @param { string } type - given data-store type (redis, inMemory etc.)
   */
  resolveDataStoreConstructor (type) {
    switch (type) {
      case 'redis':
        return RedisDataStore
      default:
        return InMemoryDataStore
    }
  }

  /**
   * Create data-store instance for given user ID
   * @param { string } userId - User ID, which uniquely identifies data-store instance
   * @param { object } options - Data-store options
   * @param { 'redis' | 'in-memory' } [options.type] - Data-store type
   */
  createDataStore (userId, { type } = {}) {
    if (this.context.has(userId)) {
      throw new Error(`Data store for user "${userId}" already exists`)
    }
    const DataStore = this.resolveDataStoreConstructor(type)
    const dataStore = new DataStore({
      keyPrefix: `${this.keyPrefix}:${userId}:`
    })
    this.context.set(userId, dataStore)
    return dataStore
  }

  /**
   * Erase data-store instance for given user ID
   * @param {string} userId - user ID
   */
  clearDataStore (userId) {
    this.context.get(userId).clear()
    this.context.delete(userId)
  }
}
