const {
  getInstructionsIfApplicable,
  processInstructions
} = require('./processing')

/**
 * @abstract
 * @class DataStore
 * @classdesc Abstract class for temporary data store
 */
module.exports = class DataStore {
  constructor ({ keyPrefix = '' } = {}) {
    if (this.constructor === DataStore) {
      throw new Error('Cannot instantiate abstract class')
    }
    this.keyPrefix = keyPrefix
    this.keys = []
  }

  /**
   * Get variable by name
   * @param { string } key - key name
   * @returns { Promise<any> } - Value recieved from the storage
   */
  async get (key) {
    const entry = await this.store.get(this.keyPrefix + key)
    return this.unpackValue(entry)
  }

  /**
   * Set variable value
   * @param { string } key - key name
   * @param { any } value - value to be kept in the storage
   * @returns { Promise<void> }
   */
  async set (key, value) {
    this.validateKey(key)
    if (!this.keys.includes(key)) this.keys.push(key)
    await this.store.set(this.keyPrefix + key, this.packValue(value))
  }

  /**
   * Delete certain data by given key
   * @param { string } key - key name
   * @returns { Promise<void> }
   */
  async delete (key) {
    await this.store.delete(this.keyPrefix + key)
    if (this.keys.includes(key)) {
      this.keys.splice(this.keys.indexOf(key), 1)
    }
  }

  /**
   * Erase all the data of this user.
   * This method can be called manually by plugin server or in response to logout event
   * to ensure we're not keeping unnecessary data
   * @returns { Promise<void> }
   */
  async clear () {
    await this.store.clear()
    this.keys = []
  }

  /**
   * Get certain data by given key and delete
   * @param { string } key - key name
   * @returns { Promise<any> } - Value recieved from the storage
   */
  async getAndDelete (key) {
    const res = await this.get(key)
    await this.delete(key)
    return res
  }

  /**
   * Process value before putting it to the storage
   * @param { any } value - value to process
   * @returns { string } - JSON encoded value with additional meta-data
   */
  packValue (value) {
    return JSON.stringify({
      value,
      postProcessing: getInstructionsIfApplicable(value)
    })
  }

  /**
   * Process value recieved from the storage
   * @param { string } packedValue - JSON encoded value from storage
   * @returns { any } - Unpacked storage data
   */
  unpackValue (packedValue) {
    if (!packedValue) return undefined

    const { value, postProcessing: instructions } = JSON.parse(packedValue)
    return instructions
      ? processInstructions(instructions, value)
      : value
  }

  /**
   * Validates given key is not empty and is of type string, throws otherwise
   * @param { string } key - key given to validate
   * @throws Will throw if argument is not string or is empty
   * @returns { void }
   */
  validateKey (key) {
    if (typeof key !== 'string') throw new Error('key parameter has to be of type string')
    if (!key) throw new Error('key parameter cannot be empty')
  }

  /**
   * Returns a proxy with fixed key prefix for current data-store instance
   * @todo Nested scoping support
   * @param { string } prefix - Prefix to scope certain method calls in
   * @returns { DataStore } - Scoped data-store instance
   */
  scope (prefix) {
    let scopedKeys = []
    const createScopedKey = key => `${prefix}:${key}`
    const proxyDataStore = new Proxy(this, {
      get (target, property) {
        switch (target[property]) {
          case target.get:
            return key => target.get(createScopedKey(key))
          case target.set:
            return (key, value) => {
              if (!scopedKeys.includes(key)) scopedKeys.push(key)
              return target.set(createScopedKey(key), value)
            }
          case target.delete:
            return key => {
              if (scopedKeys.includes(key)) {
                scopedKeys.splice(scopedKeys.indexOf(key), 1)
              }
              return target.delete(createScopedKey(key))
            }
          case target.getAndDelete:
            return target.getAndDelete.bind(proxyDataStore)
          case target.keys:
            return scopedKeys
          case target.clear:
            return () => Promise.all([...scopedKeys].map(
              key => proxyDataStore.delete(key))
            )
          default:
            return target[property]
        }
      }
    })
    return proxyDataStore
  }
}
