"use strict";
/*
 * Copyright 2021 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.JetStreamClientImpl = void 0;
const types_1 = require("./types");
const jsbaseclient_api_1 = require("./jsbaseclient_api");
const jsutil_1 = require("./jsutil");
const jsmconsumer_api_1 = require("./jsmconsumer_api");
const jsmsg_1 = require("./jsmsg");
const typedsub_1 = require("./typedsub");
const error_1 = require("./error");
const queued_iterator_1 = require("./queued_iterator");
const util_1 = require("./util");
const protocol_1 = require("./protocol");
const headers_1 = require("./headers");
const jsconsumeropts_1 = require("./jsconsumeropts");
const kv_1 = require("./kv");
var PubHeaders;
(function (PubHeaders) {
    PubHeaders["MsgIdHdr"] = "Nats-Msg-Id";
    PubHeaders["ExpectedStreamHdr"] = "Nats-Expected-Stream";
    PubHeaders["ExpectedLastSeqHdr"] = "Nats-Expected-Last-Sequence";
    PubHeaders["ExpectedLastMsgIdHdr"] = "Nats-Expected-Last-Msg-Id";
    PubHeaders["ExpectedLastSubjectSequenceHdr"] = "Nats-Expected-Last-Subject-Sequence";
})(PubHeaders || (PubHeaders = {}));
class ViewsImpl {
    constructor(js) {
        this.js = js;
        jetstreamPreview(this.js.nc);
    }
    kv(name, opts = {}) {
        return kv_1.Bucket.create(this.js, name, opts);
    }
}
class JetStreamClientImpl extends jsbaseclient_api_1.BaseApiClient {
    constructor(nc, opts) {
        super(nc, opts);
        this.api = new jsmconsumer_api_1.ConsumerAPIImpl(nc, opts);
    }
    get views() {
        return new ViewsImpl(this);
    }
    publish(subj, data = types_1.Empty, opts) {
        return __awaiter(this, void 0, void 0, function* () {
            opts = opts || {};
            opts.expect = opts.expect || {};
            const mh = (opts === null || opts === void 0 ? void 0 : opts.headers) || (0, headers_1.headers)();
            if (opts) {
                if (opts.msgID) {
                    mh.set(PubHeaders.MsgIdHdr, opts.msgID);
                }
                if (opts.expect.lastMsgID) {
                    mh.set(PubHeaders.ExpectedLastMsgIdHdr, opts.expect.lastMsgID);
                }
                if (opts.expect.streamName) {
                    mh.set(PubHeaders.ExpectedStreamHdr, opts.expect.streamName);
                }
                if (opts.expect.lastSequence) {
                    mh.set(PubHeaders.ExpectedLastSeqHdr, `${opts.expect.lastSequence}`);
                }
                if (opts.expect.lastSubjectSequence) {
                    mh.set(PubHeaders.ExpectedLastSubjectSequenceHdr, `${opts.expect.lastSubjectSequence}`);
                }
            }
            const to = opts.timeout || this.timeout;
            const ro = {};
            if (to) {
                ro.timeout = to;
            }
            if (opts) {
                ro.headers = mh;
            }
            const r = yield this.nc.request(subj, data, ro);
            const pa = this.parseJsResponse(r);
            if (pa.stream === "") {
                throw error_1.NatsError.errorForCode(error_1.ErrorCode.JetStreamInvalidAck);
            }
            pa.duplicate = pa.duplicate ? pa.duplicate : false;
            return pa;
        });
    }
    pull(stream, durable) {
        return __awaiter(this, void 0, void 0, function* () {
            (0, jsutil_1.validateStreamName)(stream);
            (0, jsutil_1.validateDurableName)(durable);
            const msg = yield this.nc.request(
            // FIXME: specify expires
            `${this.prefix}.CONSUMER.MSG.NEXT.${stream}.${durable}`, this.jc.encode({ no_wait: true, batch: 1, expires: 0 }), { noMux: true, timeout: this.timeout });
            const err = (0, jsutil_1.checkJsError)(msg);
            if (err) {
                throw (err);
            }
            return (0, jsmsg_1.toJsMsg)(msg);
        });
    }
    /*
    * Returns available messages upto specified batch count.
    * If expires is set the iterator will wait for the specified
    * amount of millis before closing the subscription.
    * If no_wait is specified, the iterator will return no messages.
    * @param stream
    * @param durable
    * @param opts
    */
    fetch(stream, durable, opts = {}) {
        (0, jsutil_1.validateStreamName)(stream);
        (0, jsutil_1.validateDurableName)(durable);
        let timer = null;
        const args = {};
        args.batch = opts.batch || 1;
        args.no_wait = opts.no_wait || false;
        if (args.no_wait && args.expires) {
            args.expires = 0;
        }
        const expires = opts.expires || 0;
        if (expires) {
            args.expires = (0, jsutil_1.nanos)(expires);
        }
        if (expires === 0 && args.no_wait === false) {
            throw new Error("expires or no_wait is required");
        }
        const qi = new queued_iterator_1.QueuedIteratorImpl();
        const wants = args.batch;
        let received = 0;
        // FIXME: this looks weird, we want to stop the iterator
        //   but doing it from a dispatchedFn...
        qi.dispatchedFn = (m) => {
            if (m) {
                received++;
                if (timer && m.info.pending === 0) {
                    // the expiration will close it
                    return;
                }
                // if we have one pending and we got the expected
                // or there are no more stop the iterator
                if (qi.getPending() === 1 && m.info.pending === 0 || wants === received) {
                    qi.stop();
                }
            }
        };
        const inbox = (0, protocol_1.createInbox)(this.nc.options.inboxPrefix);
        const sub = this.nc.subscribe(inbox, {
            max: opts.batch,
            callback: (err, msg) => {
                if (err === null) {
                    err = (0, jsutil_1.checkJsError)(msg);
                }
                if (err !== null) {
                    if (timer) {
                        timer.cancel();
                        timer = null;
                    }
                    if ((0, error_1.isNatsError)(err) &&
                        (err.code === error_1.ErrorCode.JetStream404NoMessages ||
                            err.code === error_1.ErrorCode.JetStream408RequestTimeout)) {
                        qi.stop();
                    }
                    else {
                        qi.stop(err);
                    }
                }
                else {
                    qi.received++;
                    qi.push((0, jsmsg_1.toJsMsg)(msg));
                }
            },
        });
        // timer on the client  the issue is that the request
        // is started on the client, which means that it will expire
        // on the client first
        if (expires) {
            timer = (0, util_1.timeout)(expires);
            timer.catch(() => {
                if (!sub.isClosed()) {
                    sub.drain();
                    timer = null;
                }
            });
        }
        (() => __awaiter(this, void 0, void 0, function* () {
            // close the iterator if the connection or subscription closes unexpectedly
            yield sub.closed;
            if (timer !== null) {
                timer.cancel();
                timer = null;
            }
            qi.stop();
        }))().catch();
        this.nc.publish(`${this.prefix}.CONSUMER.MSG.NEXT.${stream}.${durable}`, this.jc.encode(args), { reply: inbox });
        return qi;
    }
    pullSubscribe(subject, opts = (0, jsconsumeropts_1.consumerOpts)()) {
        return __awaiter(this, void 0, void 0, function* () {
            const cso = yield this._processOptions(subject, opts);
            if (cso.ordered) {
                throw new Error("pull subscribers cannot be be ordered");
            }
            if (!cso.attached) {
                cso.config.filter_subject = subject;
            }
            if (cso.config.deliver_subject) {
                throw new Error("consumer info specifies deliver_subject - pull consumers cannot have deliver_subject set");
            }
            const ackPolicy = cso.config.ack_policy;
            if (ackPolicy === types_1.AckPolicy.None || ackPolicy === types_1.AckPolicy.All) {
                throw new Error("ack policy for pull consumers must be explicit");
            }
            const so = this._buildTypedSubscriptionOpts(cso);
            const sub = new JetStreamPullSubscriptionImpl(this, cso.deliver, so);
            sub.info = cso;
            try {
                yield this._maybeCreateConsumer(cso);
            }
            catch (err) {
                sub.unsubscribe();
                throw err;
            }
            return sub;
        });
    }
    subscribe(subject, opts = (0, jsconsumeropts_1.consumerOpts)()) {
        return __awaiter(this, void 0, void 0, function* () {
            const cso = yield this._processOptions(subject, opts);
            // this effectively requires deliver subject to be specified
            // as an option otherwise we have a pull consumer
            if (!cso.config.deliver_subject) {
                throw new Error("consumer info specifies a pull consumer - deliver_subject is required");
            }
            const so = this._buildTypedSubscriptionOpts(cso);
            const sub = new JetStreamSubscriptionImpl(this, cso.deliver, so);
            sub.info = cso;
            try {
                yield this._maybeCreateConsumer(cso);
            }
            catch (err) {
                sub.unsubscribe();
                throw err;
            }
            return sub;
        });
    }
    _processOptions(subject, opts = (0, jsconsumeropts_1.consumerOpts)()) {
        var _a, _b;
        return __awaiter(this, void 0, void 0, function* () {
            const jsi = ((0, jsconsumeropts_1.isConsumerOptsBuilder)(opts)
                ? opts.getOpts()
                : opts);
            jsi.isBind = (0, jsconsumeropts_1.isConsumerOptsBuilder)(opts) ? opts.isBind : false;
            jsi.flow_control = {
                heartbeat_count: 0,
                fc_count: 0,
                consumer_restarts: 0,
            };
            if (jsi.ordered) {
                jsi.ordered_consumer_sequence = { stream_seq: 0, delivery_seq: 0 };
                if (jsi.config.ack_policy !== types_1.AckPolicy.NotSet &&
                    jsi.config.ack_policy !== types_1.AckPolicy.None) {
                    throw new error_1.NatsError("ordered consumer: ack_policy can only be set to 'none'", error_1.ErrorCode.ApiError);
                }
                if (jsi.config.durable_name && jsi.config.durable_name.length > 0) {
                    throw new error_1.NatsError("ordered consumer: durable_name cannot be set", error_1.ErrorCode.ApiError);
                }
                if (jsi.config.deliver_subject && jsi.config.deliver_subject.length > 0) {
                    throw new error_1.NatsError("ordered consumer: deliver_subject cannot be set", error_1.ErrorCode.ApiError);
                }
                if (jsi.config.max_deliver !== undefined && jsi.config.max_deliver > 1) {
                    throw new error_1.NatsError("ordered consumer: max_deliver cannot be set", error_1.ErrorCode.ApiError);
                }
                if (jsi.config.deliver_group && jsi.config.deliver_group.length > 0) {
                    throw new error_1.NatsError("ordered consumer: deliver_group cannot be set", error_1.ErrorCode.ApiError);
                }
                jsi.config.deliver_subject = (0, protocol_1.createInbox)(this.nc.options.inboxPrefix);
                jsi.config.ack_policy = types_1.AckPolicy.None;
                jsi.config.max_deliver = 1;
                jsi.config.flow_control = true;
                jsi.config.idle_heartbeat = jsi.config.idle_heartbeat || (0, jsutil_1.nanos)(5000);
                jsi.config.ack_wait = (0, jsutil_1.nanos)(22 * 60 * 60 * 1000);
            }
            if (jsi.config.ack_policy === types_1.AckPolicy.NotSet) {
                jsi.config.ack_policy = types_1.AckPolicy.All;
            }
            jsi.api = this;
            jsi.config = jsi.config || {};
            jsi.stream = jsi.stream ? jsi.stream : yield this.findStream(subject);
            jsi.attached = false;
            if (jsi.config.durable_name) {
                try {
                    const info = yield this.api.info(jsi.stream, jsi.config.durable_name);
                    if (info) {
                        if (info.config.filter_subject && info.config.filter_subject !== subject) {
                            throw new Error("subject does not match consumer");
                        }
                        const qn = (_a = jsi.config.deliver_group) !== null && _a !== void 0 ? _a : "";
                        const rqn = (_b = info.config.deliver_group) !== null && _b !== void 0 ? _b : "";
                        if (qn !== rqn) {
                            if (rqn === "") {
                                throw new Error(`durable requires no queue group`);
                            }
                            else {
                                throw new Error(`durable requires queue group '${rqn}'`);
                            }
                        }
                        jsi.last = info;
                        jsi.config = info.config;
                        jsi.attached = true;
                    }
                }
                catch (err) {
                    //consumer doesn't exist
                    if (err.code !== "404") {
                        throw err;
                    }
                }
            }
            if (!jsi.attached) {
                jsi.config.filter_subject = subject;
            }
            jsi.deliver = jsi.config.deliver_subject ||
                (0, protocol_1.createInbox)(this.nc.options.inboxPrefix);
            return jsi;
        });
    }
    _buildTypedSubscriptionOpts(jsi) {
        const so = {};
        so.adapter = msgAdapter(jsi.callbackFn === undefined);
        so.ingestionFilterFn = JetStreamClientImpl.ingestionFn(jsi.ordered);
        so.protocolFilterFn = (jm, ingest = false) => {
            const jsmi = jm;
            if ((0, jsutil_1.isFlowControlMsg)(jsmi.msg)) {
                if (!ingest) {
                    jsmi.msg.respond();
                }
                return false;
            }
            return true;
        };
        if (!jsi.mack && jsi.config.ack_policy !== types_1.AckPolicy.None) {
            so.dispatchedFn = autoAckJsMsg;
        }
        if (jsi.callbackFn) {
            so.callback = jsi.callbackFn;
        }
        so.max = jsi.max || 0;
        so.queue = jsi.queue;
        return so;
    }
    _maybeCreateConsumer(jsi) {
        return __awaiter(this, void 0, void 0, function* () {
            if (jsi.attached) {
                return;
            }
            if (jsi.isBind) {
                throw new Error(`unable to bind - durable consumer ${jsi.config.durable_name} doesn't exist in ${jsi.stream}`);
            }
            jsi.config = Object.assign({
                deliver_policy: types_1.DeliverPolicy.All,
                ack_policy: types_1.AckPolicy.Explicit,
                ack_wait: (0, jsutil_1.nanos)(30 * 1000),
                replay_policy: types_1.ReplayPolicy.Instant,
            }, jsi.config);
            const ci = yield this.api.add(jsi.stream, jsi.config);
            jsi.name = ci.name;
            jsi.config = ci.config;
            jsi.last = ci;
        });
    }
    static ingestionFn(ordered) {
        return (jm, ctx) => {
            // ctx is expected to be the iterator (the JetstreamSubscriptionImpl)
            const jsub = ctx;
            // this shouldn't happen
            if (!jm)
                return { ingest: false, protocol: false };
            const jmi = jm;
            if ((0, jsutil_1.isHeartbeatMsg)(jmi.msg)) {
                const ingest = ordered ? jsub._checkHbOrderConsumer(jmi.msg) : true;
                if (!ordered) {
                    jsub.info.flow_control.heartbeat_count++;
                }
                return { ingest, protocol: true };
            }
            else if ((0, jsutil_1.isFlowControlMsg)(jmi.msg)) {
                jsub.info.flow_control.fc_count++;
                return { ingest: true, protocol: true };
            }
            const ingest = ordered ? jsub._checkOrderedConsumer(jm) : true;
            return { ingest, protocol: false };
        };
    }
}
exports.JetStreamClientImpl = JetStreamClientImpl;
class JetStreamSubscriptionImpl extends typedsub_1.TypedSubscription {
    constructor(js, subject, opts) {
        super(js.nc, subject, opts);
        this.js = js;
    }
    set info(info) {
        this.sub.info = info;
    }
    get info() {
        return this.sub.info;
    }
    _resetOrderedConsumer(sseq) {
        if (this.info === null || this.sub.isClosed()) {
            return;
        }
        const newDeliver = (0, protocol_1.createInbox)(this.js.nc.options.inboxPrefix);
        const nci = this.js.nc;
        nci._resub(this.sub, newDeliver);
        const info = this.info;
        info.ordered_consumer_sequence.delivery_seq = 0;
        info.flow_control.heartbeat_count = 0;
        info.flow_control.fc_count = 0;
        info.flow_control.consumer_restarts++;
        info.deliver = newDeliver;
        info.config.deliver_subject = newDeliver;
        info.config.deliver_policy = types_1.DeliverPolicy.StartSequence;
        info.config.opt_start_seq = sseq;
        const subj = `${info.api.prefix}.CONSUMER.CREATE.${info.stream}`;
        this.js._request(subj, this.info.config)
            .catch((err) => {
            // to inform the subscription we inject an error this will
            // be at after the last message if using an iterator.
            const nerr = new error_1.NatsError(`unable to recreate ordered consumer ${info.stream} at seq ${sseq}`, error_1.ErrorCode.RequestError, err);
            this.sub.callback(nerr, {});
        });
    }
    _checkHbOrderConsumer(msg) {
        const rm = msg.headers.get(types_1.JsHeaders.ConsumerStalledHdr);
        if (rm !== "") {
            const nci = this.js.nc;
            nci.publish(rm);
        }
        const lastDelivered = parseInt(msg.headers.get(types_1.JsHeaders.LastConsumerSeqHdr), 10);
        const ordered = this.info.ordered_consumer_sequence;
        this.info.flow_control.heartbeat_count++;
        if (lastDelivered !== ordered.delivery_seq) {
            this._resetOrderedConsumer(ordered.stream_seq + 1);
        }
        return false;
    }
    _checkOrderedConsumer(jm) {
        const ordered = this.info.ordered_consumer_sequence;
        const sseq = jm.info.streamSequence;
        const dseq = jm.info.deliverySequence;
        if (dseq != ordered.delivery_seq + 1) {
            this._resetOrderedConsumer(ordered.stream_seq + 1);
            return false;
        }
        ordered.delivery_seq = dseq;
        ordered.stream_seq = sseq;
        return true;
    }
    destroy() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.isClosed()) {
                yield this.drain();
            }
            const jinfo = this.sub.info;
            const name = jinfo.config.durable_name || jinfo.name;
            const subj = `${jinfo.api.prefix}.CONSUMER.DELETE.${jinfo.stream}.${name}`;
            yield jinfo.api._request(subj);
        });
    }
    consumerInfo() {
        return __awaiter(this, void 0, void 0, function* () {
            const jinfo = this.sub.info;
            const name = jinfo.config.durable_name || jinfo.name;
            const subj = `${jinfo.api.prefix}.CONSUMER.INFO.${jinfo.stream}.${name}`;
            const ci = yield jinfo.api._request(subj);
            jinfo.last = ci;
            return ci;
        });
    }
}
class JetStreamPullSubscriptionImpl extends JetStreamSubscriptionImpl {
    constructor(js, subject, opts) {
        super(js, subject, opts);
    }
    pull(opts = { batch: 1 }) {
        var _a;
        const { stream, config, name } = this.sub.info;
        const consumer = (_a = config.durable_name) !== null && _a !== void 0 ? _a : name;
        const args = {};
        args.batch = opts.batch || 1;
        args.no_wait = opts.no_wait || false;
        if (opts.expires && opts.expires > 0) {
            args.expires = (0, jsutil_1.nanos)(opts.expires);
        }
        if (this.info) {
            const api = this.info.api;
            const subj = `${api.prefix}.CONSUMER.MSG.NEXT.${stream}.${consumer}`;
            const reply = this.sub.subject;
            api.nc.publish(subj, api.jc.encode(args), { reply: reply });
        }
    }
}
function msgAdapter(iterator) {
    if (iterator) {
        return iterMsgAdapter;
    }
    else {
        return cbMsgAdapter;
    }
}
function cbMsgAdapter(err, msg) {
    if (err) {
        return [err, null];
    }
    err = (0, jsutil_1.checkJsError)(msg);
    if (err) {
        return [err, null];
    }
    // assuming that the protocolFilterFn is set!
    return [null, (0, jsmsg_1.toJsMsg)(msg)];
}
function iterMsgAdapter(err, msg) {
    if (err) {
        return [err, null];
    }
    // iterator will close if we have an error
    // check for errors that shouldn't close it
    const ne = (0, jsutil_1.checkJsError)(msg);
    if (ne !== null) {
        switch (ne.code) {
            case error_1.ErrorCode.JetStream404NoMessages:
            case error_1.ErrorCode.JetStream408RequestTimeout:
            case error_1.ErrorCode.JetStream409MaxAckPendingExceeded:
                return [null, null];
            default:
                return [ne, null];
        }
    }
    // assuming that the protocolFilterFn is set
    return [null, (0, jsmsg_1.toJsMsg)(msg)];
}
function autoAckJsMsg(data) {
    if (data) {
        data.ack();
    }
}
const jetstreamPreview = (() => {
    let once = false;
    return (nci) => {
        var _a;
        if (!once) {
            once = true;
            const { lang } = (_a = nci === null || nci === void 0 ? void 0 : nci.protocol) === null || _a === void 0 ? void 0 : _a.transport;
            if (lang) {
                console.log(`\u001B[33m >> jetstream's materialized views functionality in ${lang} is beta functionality \u001B[0m`);
            }
            else {
                console.log(`\u001B[33m >> jetstream's materialized views functionality is beta functionality \u001B[0m`);
            }
        }
    };
})();
//# sourceMappingURL=jsclient.js.map