"use strict";
/*
 * Copyright 2021 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.isConsumerOptsBuilder = exports.ConsumerOptsBuilderImpl = exports.consumerOpts = void 0;
const types_1 = require("./types");
const jsutil_1 = require("./jsutil");
function consumerOpts(opts) {
    return new ConsumerOptsBuilderImpl(opts);
}
exports.consumerOpts = consumerOpts;
// FIXME: some items here that may need to be addressed
// 503s?
// maxRetries()
// retryBackoff()
// ackWait(time)
// replayOriginal()
// rateLimit(bytesPerSec)
class ConsumerOptsBuilderImpl {
    constructor(opts) {
        this.stream = "";
        this.mack = false;
        this.ordered = false;
        this.config = (0, jsutil_1.defaultConsumer)("", opts || {});
    }
    getOpts() {
        const o = {};
        o.config = this.config;
        o.mack = this.mack;
        o.stream = this.stream;
        o.callbackFn = this.callbackFn;
        o.max = this.max;
        o.queue = this.qname;
        o.ordered = this.ordered;
        o.config.ack_policy = o.ordered ? types_1.AckPolicy.None : o.config.ack_policy;
        o.isBind = o.isBind || false;
        return o;
    }
    description(description) {
        this.config.description = description;
    }
    deliverTo(subject) {
        this.config.deliver_subject = subject;
    }
    durable(name) {
        (0, jsutil_1.validateDurableName)(name);
        this.config.durable_name = name;
    }
    startSequence(seq) {
        if (seq <= 0) {
            throw new Error("sequence must be greater than 0");
        }
        this.config.deliver_policy = types_1.DeliverPolicy.StartSequence;
        this.config.opt_start_seq = seq;
    }
    startTime(time) {
        this.config.deliver_policy = types_1.DeliverPolicy.StartTime;
        this.config.opt_start_time = time.toISOString();
    }
    deliverAll() {
        this.config.deliver_policy = types_1.DeliverPolicy.All;
    }
    deliverLastPerSubject() {
        this.config.deliver_policy = types_1.DeliverPolicy.LastPerSubject;
    }
    deliverLast() {
        this.config.deliver_policy = types_1.DeliverPolicy.Last;
    }
    deliverNew() {
        this.config.deliver_policy = types_1.DeliverPolicy.New;
    }
    startAtTimeDelta(millis) {
        this.startTime(new Date(Date.now() - millis));
    }
    headersOnly() {
        this.config.headers_only = true;
    }
    ackNone() {
        this.config.ack_policy = types_1.AckPolicy.None;
    }
    ackAll() {
        this.config.ack_policy = types_1.AckPolicy.All;
    }
    ackExplicit() {
        this.config.ack_policy = types_1.AckPolicy.Explicit;
    }
    ackWait(millis) {
        this.config.ack_wait = (0, jsutil_1.nanos)(millis);
    }
    maxDeliver(max) {
        this.config.max_deliver = max;
    }
    filterSubject(s) {
        this.config.filter_subject = s;
    }
    replayInstantly() {
        this.config.replay_policy = types_1.ReplayPolicy.Instant;
    }
    replayOriginal() {
        this.config.replay_policy = types_1.ReplayPolicy.Original;
    }
    sample(n) {
        n = Math.trunc(n);
        if (n < 0 || n > 100) {
            throw new Error(`value must be between 0-100`);
        }
        this.config.sample_freq = `${n}%`;
    }
    limit(n) {
        this.config.rate_limit_bps = n;
    }
    maxWaiting(max) {
        this.config.max_waiting = max;
    }
    maxAckPending(max) {
        this.config.max_ack_pending = max;
    }
    idleHeartbeat(millis) {
        this.config.idle_heartbeat = (0, jsutil_1.nanos)(millis);
    }
    flowControl() {
        this.config.flow_control = true;
    }
    deliverGroup(name) {
        this.queue(name);
    }
    manualAck() {
        this.mack = true;
    }
    maxMessages(max) {
        this.max = max;
    }
    callback(fn) {
        this.callbackFn = fn;
    }
    queue(n) {
        this.qname = n;
        this.config.deliver_group = n;
    }
    orderedConsumer() {
        this.ordered = true;
    }
    bind(stream, durable) {
        this.stream = stream;
        this.config.durable_name = durable;
        this.isBind = true;
    }
    inactiveEphemeralThreshold(millis) {
        this.config.inactive_threshold = (0, jsutil_1.nanos)(millis);
    }
    maxPullBatch(n) {
        this.config.max_batch = n;
    }
    maxPullRequestExpires(millis) {
        this.config.max_expires = (0, jsutil_1.nanos)(millis);
    }
}
exports.ConsumerOptsBuilderImpl = ConsumerOptsBuilderImpl;
function isConsumerOptsBuilder(o) {
    return typeof o.getOpts === "function";
}
exports.isConsumerOptsBuilder = isConsumerOptsBuilder;
//# sourceMappingURL=jsconsumeropts.js.map