"use strict";
/*
 * Copyright 2021 The NATS Authors
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __asyncValues = (this && this.__asyncValues) || function (o) {
    if (!Symbol.asyncIterator) throw new TypeError("Symbol.asyncIterator is not defined.");
    var m = o[Symbol.asyncIterator], i;
    return m ? m.call(o) : (o = typeof __values === "function" ? __values(o) : o[Symbol.iterator](), i = {}, verb("next"), verb("throw"), verb("return"), i[Symbol.asyncIterator] = function () { return this; }, i);
    function verb(n) { i[n] = o[n] && function (v) { return new Promise(function (resolve, reject) { v = o[n](v), settle(resolve, reject, v.done, v.value); }); }; }
    function settle(resolve, reject, d, v) { Promise.resolve(v).then(function(v) { resolve({ value: v, done: d }); }, reject); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bucket = exports.validateBucket = exports.hasWildcards = exports.validateSearchKey = exports.validateKey = exports.kvOperationHdr = exports.defaultBucketOpts = exports.NoopKvCodecs = exports.Base64KeyCodec = void 0;
const types_1 = require("./types");
const jsutil_1 = require("./jsutil");
const queued_iterator_1 = require("./queued_iterator");
const util_1 = require("./util");
const headers_1 = require("./headers");
const mod_1 = require("./mod");
function Base64KeyCodec() {
    return {
        encode(key) {
            return btoa(key);
        },
        decode(bkey) {
            return atob(bkey);
        },
    };
}
exports.Base64KeyCodec = Base64KeyCodec;
function NoopKvCodecs() {
    return {
        key: {
            encode(k) {
                return k;
            },
            decode(k) {
                return k;
            },
        },
        value: {
            encode(v) {
                return v;
            },
            decode(v) {
                return v;
            },
        },
    };
}
exports.NoopKvCodecs = NoopKvCodecs;
function defaultBucketOpts() {
    return {
        replicas: 1,
        history: 1,
        timeout: 2000,
        maxBucketSize: -1,
        maxValueSize: -1,
        codec: NoopKvCodecs(),
        storage: types_1.StorageType.File,
    };
}
exports.defaultBucketOpts = defaultBucketOpts;
exports.kvOperationHdr = "KV-Operation";
const kvPrefix = "KV_";
const kvSubjectPrefix = "$KV";
const validKeyRe = /^[-/=.\w]+$/;
const validSearchKey = /^[-/=.>*\w]+$/;
const validBucketRe = /^[-\w]+$/;
// this exported for tests
function validateKey(k) {
    if (k.startsWith(".") || k.endsWith(".") || !validKeyRe.test(k)) {
        throw new Error(`invalid key: ${k}`);
    }
}
exports.validateKey = validateKey;
function validateSearchKey(k) {
    if (k.startsWith(".") || k.endsWith(".") || !validSearchKey.test(k)) {
        throw new Error(`invalid key: ${k}`);
    }
}
exports.validateSearchKey = validateSearchKey;
function hasWildcards(k) {
    if (k.startsWith(".") || k.endsWith(".")) {
        throw new Error(`invalid key: ${k}`);
    }
    const chunks = k.split(".");
    let hasWildcards = false;
    for (let i = 0; i < chunks.length; i++) {
        switch (chunks[i]) {
            case "*":
                hasWildcards = true;
                break;
            case ">":
                if (i !== chunks.length - 1) {
                    throw new Error(`invalid key: ${k}`);
                }
                hasWildcards = true;
                break;
            default:
            // continue
        }
    }
    return hasWildcards;
}
exports.hasWildcards = hasWildcards;
// this exported for tests
function validateBucket(name) {
    if (!validBucketRe.test(name)) {
        throw new Error(`invalid bucket name: ${name}`);
    }
}
exports.validateBucket = validateBucket;
class Bucket {
    constructor(bucket, jsm, js) {
        this.validateKey = validateKey;
        this.validateSearchKey = validateSearchKey;
        this.hasWildcards = hasWildcards;
        validateBucket(bucket);
        this.jsm = jsm;
        this.js = js;
        this.bucket = bucket;
        this._prefixLen = 0;
        this.subjPrefix = kvSubjectPrefix;
        const jsi = js;
        const prefix = jsi.prefix || "$JS.API";
        if (prefix !== "$JS.API") {
            this.subjPrefix = `${prefix}.${kvSubjectPrefix}`;
        }
    }
    static create(js, name, opts = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            validateBucket(name);
            const to = opts.timeout || 2000;
            const jsi = js;
            let jsopts = jsi.opts || {};
            jsopts = Object.assign(jsopts, { timeout: to });
            const jsm = yield jsi.nc.jetstreamManager(jsopts);
            const bucket = new Bucket(name, jsm, js);
            yield bucket.init(opts);
            return bucket;
        });
    }
    init(opts = {}) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            const bo = Object.assign(defaultBucketOpts(), opts);
            this.codec = bo.codec;
            const sc = {};
            this.stream = sc.name = (_a = opts.streamName) !== null && _a !== void 0 ? _a : this.bucketName();
            sc.subjects = [this.subjectForBucket()];
            sc.retention = types_1.RetentionPolicy.Limits;
            sc.max_msgs_per_subject = bo.history;
            sc.max_bytes = bo.maxBucketSize;
            sc.max_msg_size = bo.maxValueSize;
            sc.storage = bo.storage;
            sc.discard = types_1.DiscardPolicy.Old;
            sc.num_replicas = bo.replicas;
            if (bo.ttl) {
                sc.max_age = (0, jsutil_1.nanos)(bo.ttl);
            }
            sc.allow_rollup_hdrs = true;
            try {
                yield this.jsm.streams.info(sc.name);
            }
            catch (err) {
                if (err.message === "stream not found") {
                    yield this.jsm.streams.add(sc);
                }
            }
        });
    }
    bucketName() {
        var _a;
        return (_a = this.stream) !== null && _a !== void 0 ? _a : `${kvPrefix}${this.bucket}`;
    }
    subjectForBucket() {
        return `${this.subjPrefix}.${this.bucket}.>`;
    }
    subjectForKey(k) {
        return `${this.subjPrefix}.${this.bucket}.${k}`;
    }
    fullKeyName(k) {
        return `${kvSubjectPrefix}.${this.bucket}.${k}`;
    }
    get prefixLen() {
        if (this._prefixLen === 0) {
            this._prefixLen = `${kvSubjectPrefix}.${this.bucket}.`.length;
        }
        return this._prefixLen;
    }
    encodeKey(key) {
        const chunks = [];
        for (const t of key.split(".")) {
            switch (t) {
                case ">":
                case "*":
                    chunks.push(t);
                    break;
                default:
                    chunks.push(this.codec.key.encode(t));
                    break;
            }
        }
        return chunks.join(".");
    }
    decodeKey(ekey) {
        const chunks = [];
        for (const t of ekey.split(".")) {
            switch (t) {
                case ">":
                case "*":
                    chunks.push(t);
                    break;
                default:
                    chunks.push(this.codec.key.decode(t));
                    break;
            }
        }
        return chunks.join(".");
    }
    close() {
        return Promise.resolve();
    }
    dataLen(data, h) {
        const slen = h ? h.get(types_1.JsHeaders.MessageSizeHdr) || "" : "";
        if (slen !== "") {
            return parseInt(slen, 10);
        }
        return data.length;
    }
    smToEntry(key, sm) {
        return {
            bucket: this.bucket,
            key: key,
            value: sm.data,
            delta: 0,
            created: sm.time,
            revision: sm.seq,
            operation: sm.header.get(exports.kvOperationHdr) || "PUT",
            length: this.dataLen(sm.data, sm.header),
        };
    }
    jmToEntry(k, jm) {
        var _a;
        const key = this.decodeKey(jm.subject.substring(this.prefixLen));
        return {
            bucket: this.bucket,
            key: key,
            value: jm.data,
            created: new Date((0, jsutil_1.millis)(jm.info.timestampNanos)),
            revision: jm.seq,
            operation: ((_a = jm.headers) === null || _a === void 0 ? void 0 : _a.get(exports.kvOperationHdr)) || "PUT",
            delta: jm.info.pending,
            length: this.dataLen(jm.data, jm.headers),
        };
    }
    create(k, data) {
        return this.put(k, data, { previousSeq: 0 });
    }
    update(k, data, version) {
        if (version <= 0) {
            throw new Error("version must be greater than 0");
        }
        return this.put(k, data, { previousSeq: version });
    }
    put(k, data, opts = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            const ek = this.encodeKey(k);
            this.validateKey(ek);
            const o = {};
            if (opts.previousSeq !== undefined) {
                const h = (0, headers_1.headers)();
                o.headers = h;
                h.set("Nats-Expected-Last-Subject-Sequence", `${opts.previousSeq}`);
            }
            const pa = yield this.js.publish(this.subjectForKey(ek), data, o);
            return pa.seq;
        });
    }
    get(k) {
        return __awaiter(this, void 0, void 0, function* () {
            const ek = this.encodeKey(k);
            this.validateKey(ek);
            try {
                const sm = yield this.jsm.streams.getMessage(this.bucketName(), {
                    last_by_subj: this.fullKeyName(ek),
                });
                return this.smToEntry(k, sm);
            }
            catch (err) {
                if (err.message === "no message found") {
                    return null;
                }
                throw err;
            }
        });
    }
    purge(k) {
        return this._deleteOrPurge(k, "PURGE");
    }
    delete(k) {
        return this._deleteOrPurge(k, "DEL");
    }
    _deleteOrPurge(k, op) {
        var e_1, _a;
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.hasWildcards(k)) {
                return this._doDeleteOrPurge(k, op);
            }
            const iter = yield this.keys(k);
            const buf = [];
            try {
                for (var iter_1 = __asyncValues(iter), iter_1_1; iter_1_1 = yield iter_1.next(), !iter_1_1.done;) {
                    const k = iter_1_1.value;
                    buf.push(this._doDeleteOrPurge(k, op));
                    if (buf.length === 100) {
                        yield Promise.all(buf);
                        buf.length = 0;
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (iter_1_1 && !iter_1_1.done && (_a = iter_1.return)) yield _a.call(iter_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
            if (buf.length > 0) {
                yield Promise.all(buf);
            }
        });
    }
    _doDeleteOrPurge(k, op) {
        return __awaiter(this, void 0, void 0, function* () {
            const ek = this.encodeKey(k);
            this.validateKey(ek);
            const h = (0, headers_1.headers)();
            h.set(exports.kvOperationHdr, op);
            if (op === "PURGE") {
                h.set(types_1.JsHeaders.RollupHdr, types_1.JsHeaders.RollupValueSubject);
            }
            yield this.js.publish(this.subjectForKey(ek), types_1.Empty, { headers: h });
        });
    }
    _buildCC(k, history = false, opts = {}) {
        const ek = this.encodeKey(k);
        this.validateSearchKey(k);
        return Object.assign({
            "deliver_policy": history
                ? types_1.DeliverPolicy.All
                : types_1.DeliverPolicy.LastPerSubject,
            "ack_policy": types_1.AckPolicy.None,
            "filter_subject": this.fullKeyName(ek),
            "flow_control": true,
            "idle_heartbeat": (0, jsutil_1.nanos)(5 * 1000),
        }, opts);
    }
    remove(k) {
        return this.purge(k);
    }
    history(opts = {}) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            const k = (_a = opts.key) !== null && _a !== void 0 ? _a : ">";
            const qi = new queued_iterator_1.QueuedIteratorImpl();
            const done = (0, util_1.deferred)();
            const co = {};
            co.headers_only = opts.headers_only || false;
            const cc = this._buildCC(k, true, co);
            const subj = cc.filter_subject;
            const copts = (0, mod_1.consumerOpts)(cc);
            copts.orderedConsumer();
            copts.callback((err, jm) => {
                if (err) {
                    // sub done
                    qi.stop(err);
                    return;
                }
                if (jm) {
                    const e = this.jmToEntry(k, jm);
                    qi.push(e);
                    qi.received++;
                    if (jm.info.pending === 0) {
                        done.resolve();
                    }
                }
            });
            const sub = yield this.js.subscribe(subj, copts);
            done.then(() => {
                sub.unsubscribe();
            });
            done.catch((_err) => {
                sub.unsubscribe();
            });
            qi.iterClosed.then(() => {
                sub.unsubscribe();
            });
            sub.closed.then(() => {
                qi.stop();
            }).catch((err) => {
                qi.stop(err);
            });
            this.jsm.streams.getMessage(this.stream, {
                "last_by_subj": subj,
            }).catch(() => {
                // we don't have a value for this
                done.resolve();
            });
            return qi;
        });
    }
    watch(opts = {}) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            const k = (_a = opts.key) !== null && _a !== void 0 ? _a : ">";
            const qi = new queued_iterator_1.QueuedIteratorImpl();
            const co = {};
            co.headers_only = opts.headers_only || false;
            const cc = this._buildCC(k, false, co);
            const subj = cc.filter_subject;
            const copts = (0, mod_1.consumerOpts)(cc);
            copts.orderedConsumer();
            copts.callback((err, jm) => {
                if (err) {
                    // sub done
                    qi.stop(err);
                    return;
                }
                if (jm) {
                    const e = this.jmToEntry(k, jm);
                    qi.push(e);
                    qi.received++;
                }
            });
            const sub = yield this.js.subscribe(subj, copts);
            qi._data = sub;
            qi.iterClosed.then(() => {
                sub.unsubscribe();
            });
            sub.closed.then(() => {
                qi.stop();
            }).catch((err) => {
                qi.stop(err);
            });
            return qi;
        });
    }
    keys(k = ">") {
        return __awaiter(this, void 0, void 0, function* () {
            const keys = new queued_iterator_1.QueuedIteratorImpl();
            const cc = this._buildCC(k, false, { headers_only: true });
            const subj = cc.filter_subject;
            const copts = (0, mod_1.consumerOpts)(cc);
            copts.orderedConsumer();
            const sub = yield this.js.subscribe(subj, copts);
            (() => __awaiter(this, void 0, void 0, function* () {
                var e_2, _a;
                var _b;
                try {
                    for (var sub_1 = __asyncValues(sub), sub_1_1; sub_1_1 = yield sub_1.next(), !sub_1_1.done;) {
                        const jm = sub_1_1.value;
                        const op = (_b = jm.headers) === null || _b === void 0 ? void 0 : _b.get(exports.kvOperationHdr);
                        if (op !== "DEL" && op !== "PURGE") {
                            const key = this.decodeKey(jm.subject.substring(this.prefixLen));
                            keys.push(key);
                        }
                        if (jm.info.pending === 0) {
                            sub.unsubscribe();
                        }
                    }
                }
                catch (e_2_1) { e_2 = { error: e_2_1 }; }
                finally {
                    try {
                        if (sub_1_1 && !sub_1_1.done && (_a = sub_1.return)) yield _a.call(sub_1);
                    }
                    finally { if (e_2) throw e_2.error; }
                }
            }))()
                .then(() => {
                keys.stop();
            })
                .catch((err) => {
                keys.stop(err);
            });
            const si = sub;
            if (si.info.last.num_pending === 0) {
                sub.unsubscribe();
            }
            return keys;
        });
    }
    purgeBucket(opts) {
        return this.jsm.streams.purge(this.bucketName(), opts);
    }
    destroy() {
        return this.jsm.streams.delete(this.bucketName());
    }
    status() {
        var _a, _b;
        return __awaiter(this, void 0, void 0, function* () {
            const ji = this.js;
            const cluster = (_b = (_a = ji.nc.info) === null || _a === void 0 ? void 0 : _a.cluster) !== null && _b !== void 0 ? _b : "";
            const si = yield this.jsm.streams.info(this.bucketName());
            return {
                bucket: this.bucketName(),
                values: si.state.messages,
                history: si.config.max_msgs_per_subject,
                ttl: si.config.max_age,
                bucket_location: cluster,
                backingStore: si.config.storage,
            };
        });
    }
}
exports.Bucket = Bucket;
//# sourceMappingURL=kv.js.map