"use strict";
/*
* Copyright 2021 The NATS Authors
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkUnsupportedOption = exports.checkOptions = exports.parseOptions = exports.defaultOptions = void 0;
const util_1 = require("./util");
const error_1 = require("./error");
const types_1 = require("./types");
const authenticator_1 = require("./authenticator");
const transport_1 = require("./transport");
const mod_1 = require("./mod");
function defaultOptions() {
    return {
        maxPingOut: types_1.DEFAULT_MAX_PING_OUT,
        maxReconnectAttempts: types_1.DEFAULT_MAX_RECONNECT_ATTEMPTS,
        noRandomize: false,
        pedantic: false,
        pingInterval: types_1.DEFAULT_PING_INTERVAL,
        reconnect: true,
        reconnectJitter: types_1.DEFAULT_JITTER,
        reconnectJitterTLS: types_1.DEFAULT_JITTER_TLS,
        reconnectTimeWait: types_1.DEFAULT_RECONNECT_TIME_WAIT,
        tls: undefined,
        verbose: false,
        waitOnFirstConnect: false,
    };
}
exports.defaultOptions = defaultOptions;
function parseOptions(opts) {
    const dhp = `${types_1.DEFAULT_HOST}:${(0, transport_1.defaultPort)()}`;
    opts = opts || { servers: [dhp] };
    opts.servers = opts.servers || [];
    if (typeof opts.servers === "string") {
        opts.servers = [opts.servers];
    }
    if (opts.servers.length > 0 && opts.port) {
        throw new error_1.NatsError("port and servers options are mutually exclusive", error_1.ErrorCode.InvalidOption);
    }
    if (opts.servers.length === 0 && opts.port) {
        opts.servers = [`${types_1.DEFAULT_HOST}:${opts.port}`];
    }
    if (opts.servers && opts.servers.length === 0) {
        opts.servers = [dhp];
    }
    const options = (0, util_1.extend)(defaultOptions(), opts);
    // tokens don't get users
    if (opts.user && opts.token) {
        throw error_1.NatsError.errorForCode(error_1.ErrorCode.BadAuthentication);
    }
    // if authenticator, no other options allowed
    if (opts.authenticator && (opts.token || opts.user || opts.pass)) {
        throw error_1.NatsError.errorForCode(error_1.ErrorCode.BadAuthentication);
    }
    options.authenticator = (0, authenticator_1.buildAuthenticator)(options);
    ["reconnectDelayHandler", "authenticator"].forEach((n) => {
        if (options[n] && typeof options[n] !== "function") {
            throw new error_1.NatsError(`${n} option should be a function`, error_1.ErrorCode.NotFunction);
        }
    });
    if (!options.reconnectDelayHandler) {
        options.reconnectDelayHandler = () => {
            let extra = options.tls
                ? options.reconnectJitterTLS
                : options.reconnectJitter;
            if (extra) {
                extra++;
                extra = Math.floor(Math.random() * extra);
            }
            return options.reconnectTimeWait + extra;
        };
    }
    if (options.inboxPrefix) {
        try {
            (0, mod_1.createInbox)(options.inboxPrefix);
        }
        catch (err) {
            throw new error_1.NatsError(err.message, error_1.ErrorCode.ApiError);
        }
    }
    if (options.resolve) {
        if (typeof (0, transport_1.getResolveFn)() !== "function") {
            throw new error_1.NatsError(`'resolve' is not supported on this client`, error_1.ErrorCode.InvalidOption);
        }
    }
    return options;
}
exports.parseOptions = parseOptions;
function checkOptions(info, options) {
    const { proto, tls_required: tlsRequired } = info;
    if ((proto === undefined || proto < 1) && options.noEcho) {
        throw new error_1.NatsError("noEcho", error_1.ErrorCode.ServerOptionNotAvailable);
    }
    if (options.tls && !tlsRequired) {
        throw new error_1.NatsError("tls", error_1.ErrorCode.ServerOptionNotAvailable);
    }
}
exports.checkOptions = checkOptions;
function checkUnsupportedOption(prop, v) {
    if (v) {
        throw new error_1.NatsError(prop, error_1.ErrorCode.InvalidOption);
    }
}
exports.checkUnsupportedOption = checkUnsupportedOption;
//# sourceMappingURL=options.js.map