/**
 * @ignore
 */
export declare function createPair(prefix: Prefix): KeyPair;
/**
 * Creates a KeyPair with an operator prefix
 * @returns {KeyPair} Returns the created KeyPair.
 */
export declare function createOperator(): KeyPair;
/**
 * Creates a KeyPair with an account prefix
 * @returns {KeyPair} Returns the created KeyPair.
 */
export declare function createAccount(): KeyPair;
/**
 * Creates a KeyPair with an user prefix
 * @returns {KeyPair} Returns the created KeyPair.
 */
export declare function createUser(): KeyPair;
/**
 * @ignore
 */
export declare function createCluster(): KeyPair;
/**
 * @ignore
 */
export declare function createServer(): KeyPair;
/**
 * Creates a KeyPair from a specified public key
 * @param {string} Public key in string format
 * @returns {KeyPair} Returns the created KeyPair.
 * @see KeyPair#getPublicKey
 */
export declare function fromPublic(src: string): KeyPair;
/**
 * Creates a KeyPair from a specified seed.
 * @param {Uint8Array} The seed key in Uint8Array
 * @returns {KeyPair} Returns the created KeyPair.
 * @see KeyPair#getSeed
 */
export declare function fromSeed(src: Uint8Array): KeyPair;
export interface KeyPair {
    /**
     * Returns the public key associated with the KeyPair
     * @returns {string}
     * @throws NKeysError
     */
    getPublicKey(): string;
    /**
     * Returns the private key associated with the KeyPair
     * @returns Uint8Array
     * @throws NKeysError
     */
    getPrivateKey(): Uint8Array;
    /**
     * Returns the PrivateKey's seed.
     * @returns Uint8Array
     * @throws NKeysError
     */
    getSeed(): Uint8Array;
    /**
     * Returns the digital signature of signing the input with the
     * the KeyPair's private key.
     * @param {Uint8Array} input
     * @returns Uint8Array
     * @throws NKeysError
     */
    sign(input: Uint8Array): Uint8Array;
    /**
     * Returns true if the signature can be verified with the KeyPair
     * @param {Uint8Array} input
     * @param {Uint8Array} sig
     * @returns {boolean}
     * @throws NKeysError
     */
    verify(input: Uint8Array, sig: Uint8Array): boolean;
    /**
     * Clears the secret stored in the keypair. After clearing
     * a keypair cannot be used or recovered.
     */
    clear(): void;
}
/**
 * @ignore
 */
export declare enum Prefix {
    Seed = 144,
    Private = 120,
    Operator = 112,
    Server = 104,
    Cluster = 16,
    Account = 0,
    User = 160
}
/**
 * @private
 */
export declare class Prefixes {
    static isValidPublicPrefix(prefix: Prefix): boolean;
    static startsWithValidPrefix(s: string): boolean;
    static isValidPrefix(prefix: Prefix): boolean;
    static parsePrefix(v: number): Prefix;
}
/**
 * Possible error codes on exceptions thrown by the library.
 */
export declare enum NKeysErrorCode {
    InvalidPrefixByte = "nkeys: invalid prefix byte",
    InvalidKey = "nkeys: invalid key",
    InvalidPublicKey = "nkeys: invalid public key",
    InvalidSeedLen = "nkeys: invalid seed length",
    InvalidSeed = "nkeys: invalid seed",
    InvalidEncoding = "nkeys: invalid encoded key",
    InvalidSignature = "nkeys: signature verification failed",
    CannotSign = "nkeys: cannot sign, no private key available",
    PublicKeyOnly = "nkeys: no seed or private key available",
    InvalidChecksum = "nkeys: invalid checksum",
    SerializationError = "nkeys: serialization error",
    ApiError = "nkeys: api error",
    ClearedPair = "nkeys: pair is cleared"
}
export declare class NKeysError extends Error {
    name: string;
    code: string;
    chainedError?: Error;
    /**
     * @param {NKeysErrorCode} code
     * @param {Error} [chainedError]
     * @constructor
     *
     * @api private
     */
    constructor(code: NKeysErrorCode, chainedError?: Error);
}
